import { NextResponse } from "next/server";
import Stripe from "stripe";

const stripe = new Stripe(process.env.STRIPE_SECRET_KEY || "sk_test_mock_key", {
    apiVersion: "2024-12-18.acacia", // Use latest API version or mock
});

export async function POST(request: Request) {
    try {
        const { amount } = await request.json();

        // In a real app, you'd calculate amount on backend to avoid manipulation
        // For this demo/mock, we accept amount from client but normally:
        // const amount = calculateOrderAmount(items);

        const paymentIntent = await stripe.paymentIntents.create({
            amount: Math.round(amount * 100), // Convert dollars to cents
            currency: "usd",
            payment_method_types: ["card"],
        });

        return NextResponse.json({ clientSecret: paymentIntent.client_secret });
    } catch (error: any) {
        console.error("Internal Error:", error);
        // If we don't have a valid key, we can return a mock secret for testing the UI
        // specific to stripe-js testing if needed, or just error out.
        // For this "incorporated" request without keys, we might need a dummy mode.

        return NextResponse.json(
            { error: `Error creating payment intent: ${error.message}` },
            { status: 500 }
        );
    }
}
