"use client";
import React, { useState, useEffect } from "react";
import { useAuth } from "@/context/AuthContext";
import styles from "./page.module.css";
import { Package, ShoppingBag, Plus, Trash2, Edit, X, Save, MessageSquare } from "lucide-react";
import { products as initialProducts } from "@/data/products";
import { Order, Product } from "@/types";

// Mock initial orders
const initialOrders: Order[] = [
    {
        id: "ORDER-101",
        userId: "user-1",
        items: [
            { ...initialProducts[0], quantity: 1 }
        ],
        total: 6.13,
        status: "delivered",
        createdAt: "2024-12-20"
    },
    {
        id: "ORDER-102",
        userId: "user-2",
        items: [
            { ...initialProducts[2], quantity: 1 },
            { ...initialProducts[5], quantity: 2 }
        ],
        total: 93.33,
        status: "processing",
        createdAt: "2024-12-25"
    }
];

export default function AdminDashboard() {
    const { user, login } = useAuth();
    const [activeTab, setActiveTab] = useState("products");

    // Persistent State
    const [productList, setProductList] = useState<Product[]>([]);
    const [orderList, setOrderList] = useState<Order[]>([]);

    // Modal State
    const [isProductModalOpen, setIsProductModalOpen] = useState(false);
    const [newProduct, setNewProduct] = useState<Partial<Product>>({
        category: "Accessories",
        inStock: true,
        image: "https://via.placeholder.com/300"
    });

    // State for Editing
    const [editingProductId, setEditingProductId] = useState<string | null>(null);

    // State for Login
    const [loginEmail, setLoginEmail] = useState("admin@chigoexpress.com");
    const [loginPass, setLoginPass] = useState("admin123");
    const [isLoggingIn, setIsLoggingIn] = useState(false);

    // 1. Load Data on Mount
    useEffect(() => {
        // CHANGED: Use '_v2' keys to force-load the new data with descriptions
        const savedProducts = localStorage.getItem("admin_products_v2");
        const savedOrders = localStorage.getItem("admin_orders_v2");

        if (savedProducts) {
            setProductList(JSON.parse(savedProducts));
        } else {
            // Initial Load: Use the enriched static data
            setProductList(initialProducts);
            localStorage.setItem("admin_products_v2", JSON.stringify(initialProducts));
        }

        if (savedOrders) {
            setOrderList(JSON.parse(savedOrders));
        } else {
            setOrderList(initialOrders);
            localStorage.setItem("admin_orders_v2", JSON.stringify(initialOrders));
        }
    }, []);

    // 2. Save Data Helper
    const saveProducts = (updatedProducts: Product[]) => {
        setProductList(updatedProducts);
        localStorage.setItem("admin_products_v2", JSON.stringify(updatedProducts));
    };

    const deleteProduct = (id: string) => {
        if (confirm("Are you sure you want to delete this product?")) {
            saveProducts(productList.filter(p => p.id !== id));
        }
    };

    const handleEditClick = (product: Product) => {
        setEditingProductId(product.id);
        setNewProduct(product); // Populate form with product data
        setIsProductModalOpen(true);
    };

    const handleSaveProduct = (e: React.FormEvent) => {
        e.preventDefault();

        if (editingProductId) {
            // UPDATE Existing Product
            const updatedList = productList.map(p =>
                p.id === editingProductId
                    ? { ...p, ...newProduct } as Product
                    : p
            );
            saveProducts(updatedList);
            alert("Product Updated Successfully!");
        } else {
            // CREATE New Product
            const product: Product = {
                id: Date.now().toString(),
                name: newProduct.name || "New Product",
                price: Number(newProduct.price) || 0,
                category: newProduct.category as any || "Others",
                image: newProduct.image || "https://via.placeholder.com/300",
                description: newProduct.description || "",
                inStock: true
            };
            saveProducts([product, ...productList]);
            alert("Product Added Successfully!");
        }

        // Reset Modal
        setIsProductModalOpen(false);
        setEditingProductId(null);
        setNewProduct({ category: "Accessories", inStock: true, image: "https://via.placeholder.com/300" });
    };

    const handleAdminLogin = async (e: React.FormEvent) => {
        e.preventDefault();
        setIsLoggingIn(true);
        try {
            await login(loginEmail, loginPass);
            // Page will re-render automatically due to auth state change
        } catch (err) {
            alert("Login Failed. Please check credentials.");
        } finally {
            setIsLoggingIn(false);
        }
    };

    const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            // Check size (limit to 2MB to save LocalStorage space)
            if (file.size > 2 * 1024 * 1024) {
                alert("File size too large! Please enter an image URL instead or use a smaller image (< 2MB).");
                return;
            }

            const reader = new FileReader();
            reader.onloadend = () => {
                setNewProduct({ ...newProduct, image: reader.result as string });
            };
            reader.readAsDataURL(file);
        }
    };

    if (!user || user.role !== "admin") {
        return (
            <div className={styles.container}>
                <div style={{ maxWidth: '600px', margin: '4rem auto', textAlign: 'center' }}>
                    <h1 style={{ fontSize: '2.5rem', marginBottom: '1rem' }}>Admin Login</h1>
                    <p style={{ marginBottom: '2.5rem', fontSize: '1.1rem' }}>Please sign in to access the dashboard.</p>

                    <form onSubmit={handleAdminLogin} style={{ display: 'flex', flexDirection: 'column', gap: '1.5rem', background: '#f5f5f5', padding: '3rem', borderRadius: '12px', boxShadow: '0 4px 20px rgba(0,0,0,0.05)' }}>
                        <div style={{ textAlign: 'left' }}>
                            <label style={{ display: 'block', marginBottom: '0.8rem', fontWeight: 'bold', fontSize: '1.1rem' }}>Email</label>
                            <input
                                type="email"
                                value={loginEmail}
                                onChange={e => setLoginEmail(e.target.value)}
                                style={{ width: '100%', padding: '1rem', borderRadius: '8px', border: '1px solid #ccc', fontSize: '1.1rem' }}
                            />
                        </div>
                        <div style={{ textAlign: 'left' }}>
                            <label style={{ display: 'block', marginBottom: '0.8rem', fontWeight: 'bold', fontSize: '1.1rem' }}>Password</label>
                            <input
                                type="password"
                                value={loginPass}
                                onChange={e => setLoginPass(e.target.value)}
                                style={{ width: '100%', padding: '1rem', borderRadius: '8px', border: '1px solid #ccc', fontSize: '1.1rem' }}
                            />
                        </div>
                        <button
                            type="submit"
                            disabled={isLoggingIn}
                            style={{
                                background: '#E65100', // Brand Orange
                                color: 'white',
                                padding: '1.2rem',
                                border: 'none',
                                borderRadius: '8px',
                                fontWeight: 'bold',
                                fontSize: '1.1rem',
                                cursor: 'pointer',
                                marginTop: '1.5rem',
                                transition: 'background 0.2s'
                            }}
                        >
                            {isLoggingIn ? "Signing In..." : "Sign In to Dashboard"}
                        </button>
                    </form>

                    <div style={{ marginTop: '2rem', fontSize: '0.9rem', color: '#666', background: '#fff', padding: '1rem', border: '1px dashed #ccc' }}>
                        <strong>Default Credentials:</strong><br />
                        Email: admin@chigoexpress.com<br />
                        Password: admin123
                    </div>
                </div>
            </div>
        );
    }

    return (
        <div className={styles.container}>
            <div className={styles.sidebar}>
                <div className={styles.profile}>
                    <div className={styles.avatar}>{user.name.charAt(0).toUpperCase()}</div>
                    <div>
                        <div className={styles.name}>{user.name}</div>
                        <div className={styles.role}>Administrator</div>
                    </div>
                </div>
                <nav className={styles.nav}>
                    <button
                        className={`${styles.navItem} ${activeTab === "products" ? styles.active : ""}`}
                        onClick={() => setActiveTab("products")}
                    >
                        <Package size={20} /> Products
                    </button>
                    <button
                        className={`${styles.navItem} ${activeTab === "orders" ? styles.active : ""}`}
                        onClick={() => setActiveTab("orders")}
                    >
                        <ShoppingBag size={20} /> Orders
                    </button>
                    <button
                        className={`${styles.navItem} ${activeTab === "complaints" ? styles.active : ""}`}
                        onClick={() => setActiveTab("complaints")}
                    >
                        <MessageSquare size={20} /> Complaints
                    </button>
                </nav>
            </div>

            <main className={styles.content}>
                <header className={styles.header}>
                    <h1>{activeTab.charAt(0).toUpperCase() + activeTab.slice(1)} Management</h1>
                    {activeTab === "products" && (
                        <button
                            className={styles.addBtn}
                            onClick={() => {
                                setEditingProductId(null);
                                setNewProduct({ category: "Accessories", inStock: true, image: "https://via.placeholder.com/300" });
                                setIsProductModalOpen(true);
                            }}
                        >
                            <Plus size={20} /> Add Product
                        </button>
                    )}
                </header>

                {activeTab === "products" && (
                    <div className={styles.tableWrapper}>
                        <table className={styles.table}>
                            <thead>
                                <tr>
                                    <th>Image</th>
                                    <th>Name</th>
                                    <th>Category</th>
                                    <th>Price</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                {productList.map(product => (
                                    <tr key={product.id}>
                                        <td>
                                            <img src={product.image} alt="" className={styles.thumb} />
                                        </td>
                                        <td>
                                            <div className={styles.productName}>{product.name}</div>
                                            <div className={styles.productDesc}>{product.description.substring(0, 50)}...</div>
                                        </td>
                                        <td>{product.category}</td>
                                        <td>${product.price ? product.price.toFixed(2) : "0.00"}</td>
                                        <td>
                                            <div className={styles.actions}>
                                                <button className={styles.actionBtn} onClick={() => handleEditClick(product)}>
                                                    <Edit size={16} />
                                                </button>
                                                <button className={styles.deleteBtn} onClick={() => deleteProduct(product.id)}>
                                                    <Trash2 size={16} />
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                )}

                {activeTab === "orders" && (
                    <div className={styles.tableWrapper}>
                        <table className={styles.table}>
                            <thead>
                                <tr>
                                    <th>Order ID</th>
                                    <th>Date</th>
                                    <th>Items</th>
                                    <th>Total</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                {orderList.map(order => (
                                    <tr key={order.id}>
                                        <td className={styles.orderId}>{order.id}</td>
                                        <td>{order.createdAt}</td>
                                        <td>
                                            {order.items.map(i => (
                                                <div key={i.id}>{i.quantity}x {i.name.substring(0, 20)}...</div>
                                            ))}
                                        </td>
                                        <td className={styles.price}>${order.total.toFixed(2)}</td>
                                        <td>
                                            <span className={`${styles.statusBadge} ${styles[order.status]}`}>
                                                {order.status}
                                            </span>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                )}

                {activeTab === "complaints" && (
                    <div className={styles.placeholder}>
                        <p>No customer complaints found.</p>
                    </div>
                )}
            </main>

            {/* ADD PRODUCT MODAL */}
            {isProductModalOpen && (
                <div className={styles.modalOverlay}>
                    <div className={styles.modal}>
                        <div className={styles.modalHeader}>
                            <h2>{editingProductId ? "Edit Product" : "Add New Product"}</h2>
                            <button onClick={() => { setIsProductModalOpen(false); setEditingProductId(null); }}><X size={24} /></button>
                        </div>
                        <form onSubmit={handleSaveProduct} className={styles.modalForm}>
                            <div className={styles.formGroup}>
                                <label>Product Name</label>
                                <input
                                    required
                                    value={newProduct.name || ""}
                                    onChange={e => setNewProduct({ ...newProduct, name: e.target.value })}
                                    placeholder="e.g. Wireless Headphones"
                                />
                            </div>
                            <div className={styles.row}>
                                <div className={styles.formGroup}>
                                    <label>Price ($)</label>
                                    <input
                                        type="number"
                                        required
                                        value={newProduct.price || ""}
                                        onChange={e => setNewProduct({ ...newProduct, price: parseFloat(e.target.value) })}
                                        placeholder="0.00"
                                    />
                                </div>
                                <div className={styles.formGroup}>
                                    <label>Category</label>
                                    <select
                                        value={newProduct.category}
                                        onChange={e => setNewProduct({ ...newProduct, category: e.target.value as any })}
                                    >
                                        <option>Accessories</option>
                                        <option>Audio</option>
                                        <option>Smart Devices</option>
                                        <option>Others</option>
                                    </select>
                                </div>
                            </div>

                            {/* IMAGE UPLOAD SECTION */}
                            <div className={styles.formGroup}>
                                <label>Product Image</label>
                                <div style={{ display: 'flex', flexDirection: 'column', gap: '0.5rem' }}>
                                    <input
                                        type="file"
                                        accept="image/*"
                                        onChange={handleImageUpload}
                                        style={{ border: '1px dashed #ccc', padding: '1rem' }}
                                    />
                                    <div style={{ textAlign: 'center', fontSize: '0.8rem', color: '#888' }}>- OR -</div>
                                    <input
                                        value={newProduct.image || ""}
                                        onChange={e => setNewProduct({ ...newProduct, image: e.target.value })}
                                        placeholder="Paste Image URL (https://...)"
                                    />
                                </div>
                                {newProduct.image && (
                                    <div style={{ marginTop: '0.5rem' }}>
                                        <p style={{ fontSize: '0.8rem', marginBottom: '0.25rem' }}>Preview:</p>
                                        <img
                                            src={newProduct.image}
                                            alt="Preview"
                                            style={{ width: '80px', height: '80px', objectFit: 'cover', borderRadius: '4px', border: '1px solid #eee' }}
                                        />
                                    </div>
                                )}
                            </div>

                            <div className={styles.formGroup}>
                                <label>Description</label>
                                <textarea
                                    rows={4}
                                    value={newProduct.description || ""}
                                    onChange={e => setNewProduct({ ...newProduct, description: e.target.value })}
                                    placeholder="Product description..."
                                />
                            </div>
                            <button type="submit" className={styles.saveBtn}>
                                <Save size={18} /> {editingProductId ? "Update Product" : "Save Product"}
                            </button>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
